#include "vertexShaderSemantics.fx"
#include "fragmentShaderSemantics.fx"

#ifdef _FEATHERS
sampler2D diffuse_tex		: register(s0); // Individual feather diffuse texture
sampler2D normalmap_tex		: register(s1); // Individual feather normal map texture
sampler2D specular_tex		: register(s2); // Individual feather specular texture
sampler2D surf_diff_tex		: register(s3); // Surface/body diffuse texture
sampler2D surf_spec_tex		: register(s4); // Surface/body specular texture
#else
sampler2D base_diff_tex		: register(s0); // Base surface diffuse texture
sampler2D normalmap_tex		: register(s1); // Fur normal map texture
sampler2D specular_tex		: register(s2); // Individual hair specular texture
sampler2D tip_diff_tex		: register(s3); // Tip surface diffuse texture
sampler2D surf_spec_tex		: register(s4); // Surface/body specular texture
#endif


sampler2D depth_buffer		: register(s5); // Depth texture

samplerCUBE diffenv_tex		: register(s6); // Diffuse environment cubemap

float3	kBaseColour			: register(c202);
float3	kTipColour			: register(c203);
float3	kBaseAmbientColour	: register(c204);
float3	kTipAmbientColour	: register(c205);
float3	kInputSpecularColour : register(c206);

float4	VSParams			: register(c200); // (Segments, Feathers, BaseAlpha, TipAlpha )
float4	VSParams2			: register(c201); // (AntiAliasAlphaMul, NumInstances, ?, FadeDepth)

float4	PSParams			: register(c211); // (Specular Sharpness, Edga Alpha, Translucency, Translucency Spread ) 
float4	PSParams2			: register(c212); // (Self Shadowing, furDiffuseEnvMapDiffIntensity, furDiffuseEnvMapSpecIntensity, furDiffuseEnvMapFresnelPower)
float4	PSParams3			: register(c213); // (MinRange, MaxRange, FallOffDist, FallOffDist)
float4  PSParams4			: register(c219); // (Feathers, ?, ?, ?)

float4x4 envMtx				: register(c214); // Diffuse environment rotation matrix (view matrix).

float4  PSReflectParams     : register(c218); // (reflectionFadeDepth, reflectionFadePower, reflectionMaxIntensity, ?)

struct FragmentOutput 
{
	float4 Colour : Color0;
	float4 normal : Color1;
	float4 albedo : Color2;
#ifdef _PC_TARGET
	float4 depth: Color3;
#endif
};

struct VS_OUT
{
	float4 ProjPos  : POSITION;		// Projected space position 
	float3 Tangent	: TEXCOORD0;	// Tangent direction (eye space, because thats where the lights are)
	float4 Bin		: TEXCOORD1;	// xyz = Binormal direction (eye space), w = vertex world y
	float3 View		: TEXCOORD2;	// View vector (in eye space)
	float3 Nrm		: TEXCOORD3;	// Normal vector of underlying surface
	float4 Params	: TEXCOORD4;	// Hair parameters (actually t,side,ldotn,1-t)
	float4 Params2	: TEXCOORD5;	// More hair parameters (U, V, base width, tip width)
	float4 Screen	: TEXCOORD6;	// Projected position (for use in PS)
};

#ifndef _PC_TARGET
/*
VS_OUT surfgeom_360_vx( int Index : INDEX )
{
	VS_OUT Out;
	
	float2 ScreenSize = vs_screenSize.xy;
	
	float kFeathers = VSParams.y;
	float kTipAlpha = VSParams.w;
	float kBaseAlpha = VSParams.z;
	float kAntialiasAlphaMul = VSParams2.x;
	int Segments = VSParams.x;
	//int SegmentVerts = (Segments+1)*2; // Width
	//int TotalNumVerts = SegmentVerts+2; // 2 Extra for tri-stripping
	
	// Calculate index of hair
	//int instanceIndex = (Index+0.5) / TotalNumVerts;
	
	//float a = TotalNumVerts*instanceIndex;
	//float b = Index - a;
	//float c = (b+0.5) / TotalNumVerts;
	//instanceIndex += c;
	
	int polyIndex = Index / 6;
	float4 SortIndex;
	
	// Fetch sorted index buffer
	asm 
	{
		vfetch SortIndex, polyIndex, position5
	};
	
	int vertIndex = Index - polyIndex*6;
	int instanceIndex = SortIndex.x;
	int segmentIndex = SortIndex.y;
	
	float4 CP1,CP2,CP3,CP4;
	float4 Nrm; // (nrm.xyz,dist)

	// Fetch control points
	asm 
	{
		vfetch CP1, instanceIndex, position0
		vfetch CP2, instanceIndex, position1
		vfetch CP3, instanceIndex, position2
		vfetch CP4, instanceIndex, position3
		vfetch Nrm, instanceIndex, position4
	};
	
	Nrm.z = sqrt( 1.0 - Nrm.x*Nrm.x + Nrm.y*Nrm.y );
	
	float vt = (clamp(vertIndex,1,4)-1) % 2; 	
	float t = (float(segmentIndex) + vt) / (float)Segments;
	float it = 1.0 - t;
		
	float3 curvePos = CP1.xyz*(it*it*it) + 3.0*CP2.xyz*(t*it*it) + 3.0*CP3.xyz*(t*t*it) + CP4.xyz*(t*t*t);
	
	// Flips the side each of the main hair verts is on
	float side = (vertIndex % 2) - 0.5;
	
	// Get tangent vector
	float t2 = t*t;
	float3 tangent = CP1.xyz*3.0*(2.0*t-1.0-t2) + CP2.xyz*3.0*(1.0-4.0*t+3.0*t2) + CP3.xyz*3.0*(2.0*t - 3.0*t2) + CP4.xyz*3.0*t2;
	tangent = normalize(tangent);
	
	// Add face-on/surface-tangent vector to give width (based on feather parameter)
	float3 viewVec = normalize(worldCamPos - curvePos) * (1-kFeathers) + normalize(Nrm.xyz) * kFeathers;
	float3 widthVec = normalize( cross( viewVec, tangent ) );
		
	//float width = (side * 0.090 * it); // xit just makes the hair thinner at tip
	float width = side * (CP3.w * it + CP4.w * t); // Calculate width from base and tip widths (stored in control point w coords)
	float3 vertexPos = curvePos - widthVec * width; 

	Out.ProjPos = mul(float4(vertexPos,1), viewProj);
	float4 ScreenCurvePos = mul(float4(curvePos,1), viewProj);
	
	// Calculate alpha based on screen space dimensions of hair (i.e. 50% pixel coverage = 50% alpha)
	float2 size = Out.ProjPos.xy/Out.ProjPos.w - ScreenCurvePos.xy/ScreenCurvePos.w;
	size *= ScreenSize * 0.5; //  Projection space is -1->1
	float pixelWidth = max( length(size), 0.0001 ); // Limit rendering to hairs as small as one 10,000th of a pixel
	float alpha = clamp( pixelWidth * kAntialiasAlphaMul, 0, 1 );
	alpha = (kTipAlpha*t + kBaseAlpha*it) * alpha; // Scale up just so that results aren't too smooth (some noise is good!)
	
	// Adjust position to make sure every hair is at least one pixel thick
	vertexPos = curvePos - widthVec * (width * (1 / min(pixelWidth,1))); 
	Out.ProjPos = mul(float4(vertexPos,1), viewProj);
		
	// Output other data
	Out.Tangent = mul(tangent, vs_view);
	Out.Bin = float4(mul(widthVec, vs_view), 0.0);
	Out.View = mul(vertexPos, vs_view);
	Out.Nrm = mul(Nrm.xyz, vs_view);
	Out.Params = float4( t, side*2, 1.0f, alpha );
	Out.Params2 = float4( CP1.w, CP2.w, CP3.w, CP4.w );
	Out.Screen = Out.ProjPos.xyw;
	
	return Out;
}
*/

VS_OUT surfgeom_360_vx( int Index : INDEX )
{
	VS_OUT Out;
	
	float2 ScreenSize = vs_screenSize.xy;
	
	float kFeathers = VSParams.y;
	float kTipAlpha = VSParams.w;
	float kBaseAlpha = VSParams.z;
	float kAntialiasAlphaMul = VSParams2.x;
	float NumInstances = VSParams2.y;
	
	int Segments = VSParams.x;
	int SegmentVerts = (Segments+1)*2; // Width
	int TotalNumVerts = SegmentVerts+2; // 2 Extra for tri-stripping
	
	// Calculate index of hair
	int instanceIndex = (Index+0.5) / TotalNumVerts;
	
	// Required to ensure instanceIndex is accurate for very large numbers of hairs
	// (possibly caused by innacuracies in the vertex maths processor?)
	float a = TotalNumVerts*instanceIndex;
	float b = Index - a;
	float c = (b+0.5) / TotalNumVerts;
	instanceIndex += c;
	
	float4 CP1,CP2,CP3,CP4;
	float4 Nrm; // (nrm.xyz,dist)

	// Fetch control points
	asm 
	{
		vfetch CP1, instanceIndex, position0
		vfetch CP2, instanceIndex, position1
		vfetch CP3, instanceIndex, position2
		vfetch CP4, instanceIndex, position3
		vfetch Nrm, instanceIndex, position4
	};
	
	int vertNum = (float(Index)+0.5 - TotalNumVerts*float(instanceIndex));
	float t = clamp( float(vertNum - 2) / (Segments*2-1), 0.0, 1.0 );
	float it = 1.0 - t;
	float t2 = t*t;

	/*float2 tvals = float2(t,it);
	float4 params = tvals.yyxx;
	params *= tvals.yyxx;
	params *= tvals.yxyx;
	params.yz *= 3.0;
	float3 curvePos = CP1.xyz*params.x + CP2.xyz*params.y + CP3.xyz*params.z + CP4.xyz*params.w;*/
	float3 curvePos = CP1.xyz*(it*it*it) + 3.0*CP2.xyz*(t*it*it) + 3.0*CP3.xyz*(t*t*it) + CP4.xyz*(t*t*t);
	
	// Get tangent vector
	float4 params = float4(6,-12,6,0)*t;
	params += float4(-3,9,-9,3)*t2;
	params.xy += float2(-3,3);
	float3 tangent = CP1.xyz*params.x + CP2.xyz*params.y + CP3.xyz*params.z + CP4.xyz*params.w;
	//float3 tangent = CP1.xyz*3.0*(2.0*t-1.0-t2) + CP2.xyz*3.0*(1.0-4.0*t+3.0*t2) + CP3.xyz*3.0*(2.0*t - 3.0*t2) + CP4.xyz*3.0*t2;
	tangent = normalize(tangent);
	
	// Add face-on/surface-tangent vector to give width (based on feather parameter)
	float3 toCam = worldCamPos - curvePos;
	float CamDist = length(toCam);
	float3 viewVec = normalize(toCam) * (1-kFeathers) + normalize(Nrm.xyz) * kFeathers;
	float3 widthVec = normalize( cross( viewVec, tangent ) );
		
	// Flips the side each of the main hair verts is on
	float side = ( clamp(vertNum,1,SegmentVerts) % 2) - 0.5;

	// Calculate width from base and tip widths (stored in control point w coords)
	float width = side * (CP3.w * it + CP4.w * t); 

	// Calculate local space vertex position
	float3 widthOffset = widthVec * width;
	float3 vertexPos = curvePos - widthOffset; 

	// Calculate screen space vertex position and curve position (center of hair)
	Out.ProjPos = mul(float4(vertexPos,1), viewProj);
	float4 ScreenCurvePos = mul(float4(curvePos,1), viewProj);
	
	// Calculate alpha based on screen space dimensions of hair (i.e. 50% pixel coverage = 50% alpha)
	float2 size = Out.ProjPos.xy/Out.ProjPos.w - ScreenCurvePos.xy/ScreenCurvePos.w;
	size *= ScreenSize * 0.5; //  Projection space is -1->1
	float pixelWidth = max( length(size), 0.0001 ); // Limit rendering to hairs as small as one 10,000th of a pixel
	float alpha = clamp( pixelWidth * kAntialiasAlphaMul, 0, 1 ); // Modulate AA alpha effect by constant user-setting
	alpha = (kTipAlpha*t + kBaseAlpha*it) * alpha;
	
	// Adjust position to make sure every hair is at least one pixel thick
	vertexPos = curvePos - widthOffset * (1 / min(pixelWidth,1)); 
	Out.ProjPos = mul(float4(vertexPos,1), viewProj);
		
	// Output other data
	Out.Tangent = mul(tangent, vs_view);
	Out.Bin = float4( mul(widthVec, (float3x3)vs_view), vertexPos.y );
	Out.View = mul(vertexPos, vs_view);
	Out.Nrm = mul(Nrm.xyz, vs_view);
	Out.Params = float4( t, side*2, 1.0f, alpha );
	Out.Params2 = float4( CP1.w, CP2.w, CP3.w, CamDist );
	Out.Screen = Out.ProjPos;
	
	return Out;
}
#endif

struct VS_IN_PC
{
	float4 VertParams	: POSITION0; // (t, side, ?, ?)
	float4 CP1			: POSITION1; // w = U
	float4 CP2			: POSITION2; // w = V
	float4 CP3			: POSITION3; // w = base width
	float4 CP4			: POSITION4; // w = tip width
	float4 Nrm			: POSITION5; // (nrm.x, nrm.y, nrm.z, dist)
};


VS_OUT surfgeom_PC_vx( VS_IN_PC In )
{
	VS_OUT Out;

	// If the fur tip is below reflected floor fade depth cutoff then immediately return an off-screen value.
	float height = dot( In.CP4.y, world[1] );
	if (height < -VSParams2.w)
	{
		float4 zero = float4(0.0, 0.0, 0.0, 0.0);
		float4 one  = float4(1.0, 0.0, 0.0, 1.0);
		float4 far  = float4(In.CP4.x - 10000.0, In.CP4.y - 10000.0, In.CP4.z - 10000.0, 1.0);
		Out.ProjPos = far;
		Out.Tangent = one.yzxw;
		Out.Bin     = one.zxyw;
		Out.View    = one;
		Out.Nrm     = one;
		Out.Params  = zero;
		Out.Params2 = zero;
		Out.Screen  = far;
		return Out;
	}
	else
	{
		float2 ScreenSize = vs_screenSize.xy;
		
		float kFeathers = VSParams.y;
		float kTipAlpha = VSParams.w;
		float kBaseAlpha = VSParams.z;
		float kAntialiasAlphaMul = VSParams2.x;
		
		float t = In.VertParams.x;
		float it = 1.0 - t;
		float t2 = t*t;
		float it2 = it*it;

		float3 curvePos = In.CP1.xyz*(it*it2) + 3.0*In.CP2.xyz*(t*it2) + 3.0*In.CP3.xyz*(t2*it) + In.CP4.xyz*(t2*t);
		
		// Flips the side each of the main hair verts is on
		float side = In.VertParams.y;
		
		// Get tangent vector
		float4 params = float4(6,-12,6,0)*t;
		params += float4(-3,9,-9,3)*t2;
		params.xy += float2(-3,3);
		float3 tangent = In.CP1.xyz*params.x + In.CP2.xyz*params.y + In.CP3.xyz*params.z + In.CP4.xyz*params.w;
		//float3 tangent = In.CP1.xyz*3.0*(2.0*t-1.0-t2) + In.CP2.xyz*3.0*(1.0-4.0*t+3.0*t2) + In.CP3.xyz*3.0*(2.0*t - 3.0*t2) + In.CP4.xyz*3.0*t2;
		tangent = normalize(tangent);

		// Add face-on/surface-tangent vector to give width (based on feather parameter)
		float3 toCam = worldCamPos - curvePos;
		float CamDist = length(toCam);
		float3 toCamNorm = toCam / CamDist;
		float3 viewVec = toCamNorm * (1-kFeathers) + normalize(In.Nrm.xyz) * kFeathers;
		float3 widthVec = normalize( cross( viewVec, tangent ) );
		
		//float EdgeOn = pow( abs( dot( toCamNorm, widthVec ) ), 2 );
		//float3 curveFaceOnPos = In.CP1.xyz*it + In.CP4.xyz*t;
		//curvePos = lerp( curveFaceOnPos, curvePos, EdgeOn );

		float width = side * (In.CP3.w * it + In.CP4.w * t); // Calculate width from base and tip widths (stored in control point w coords)
		float3 vertexPos = curvePos - widthVec * width; 
		
		Out.ProjPos = mul(float4(vertexPos,1), viewProj);
		float4 ScreenCurvePos = mul(float4(curvePos,1), viewProj);
		
		// Calculate alpha based on screen space dimensions of hair (i.e. 50% pixel coverage = 50% alpha)
		float2 size = Out.ProjPos.xy/Out.ProjPos.w - ScreenCurvePos.xy/ScreenCurvePos.w;
		size *= ScreenSize * 0.5; //  Projection space is -1->1
		float pixelWidth = max( length(size), 0.0001 ); // Limit rendering to hairs as small as one 10,000th of a pixel
		float alpha = clamp( pixelWidth * kAntialiasAlphaMul, 0, 1 );
		alpha = (kTipAlpha*t + kBaseAlpha*it) * alpha; // Scale up just so that results aren't too smooth (some noise is good!)
		
		// Adjust position to make sure every hair is at least one pixel thick
		float pixelWidthMul = lerp( (1 / min(pixelWidth,1)), 1, kFeathers );

		vertexPos = curvePos - widthVec * (width * pixelWidthMul);
		float4 worldVertexPos = mul(float4(vertexPos,1), world);

		Out.ProjPos = mul(worldVertexPos, viewProj);     // REFLECTION_CODE (also remove above line)
		Out.Tangent = mul(tangent,    (float3x3)vs_view);
		Out.Bin     = float4(mul(widthVec, (float3x3)vs_view), worldVertexPos.y);
		Out.View    = mul(worldVertexPos,  vs_view).xyz;
		Out.Nrm     = mul(In.Nrm.xyz, (float3x3)vs_view);
		Out.Params  = float4( t, side*2, 1.0f, alpha );
		Out.Params2 = float4( In.CP1.w, In.CP2.w, In.CP3.w, CamDist );
		Out.Screen  = Out.ProjPos;

		return Out;
	}	
}

FragmentOutput surfgeom_px( VS_OUT In )
{
	FragmentOutput fout;

	// Clip out any normal geometry passing downwards through the floor, and also any reflected geometry passing upwards through the floor.
	float worldY = In.Bin.w;
	clip(worldY * fs_lightingAdjust.y);
	
	// Get screen space coordinates
	float3 ScreenPos = (In.Screen.xyz / In.Screen.www) * float3(0.5,0.5,1.0);
	ScreenPos.y = -ScreenPos.y;
	ScreenPos.xy += 0.5;
	
	// Add a 0.5 texel offset (DX9 compensation)
	ScreenPos.xy += 0.5 * fs_screenSize.zw;

	// Get projection space depths
	float2 Depths;
#ifndef _PC_TARGET // Depths( x = front, y = current )
	Depths.x = 1-tex2D( depth_buffer, ScreenPos.xy ).r;
	Depths.y = 1-(In.Screen.z / In.Screen.w);
	
	// Transform depths to world space
	Depths = fs_projection_params.xx / (float2(1,1) - Depths*fs_projection_params.ww);
	
#else
	Depths.x = tex2D( depth_buffer, ScreenPos.xy ).r * 1000.0;
	//Depths.x = tex2Dlod( depth_buffer, float4(ScreenPos.x,ScreenPos.x,0,0) ).r * 10.0;
	Depths.y = (In.Screen.z / In.Screen.w);

	//Depths = fs_projection_params.xx / (float2(1,1) - Depths*fs_projection_params.ww);
	Depths.y = fs_projection_params.x / (1.0 - Depths.y*fs_projection_params.w);
#endif

	// Calculate depth extents of current pass
	float2 MinMaxDepths = PSParams3.xy + Depths.xx;

	// Calculate distance from each bounding plane for current pass
	float2 PlaneDists = float2(Depths.y - MinMaxDepths.x, MinMaxDepths.y - Depths.y) * PSParams3.zw + 0.001; 
	
	// Calculate alpha and clip transparent pixels
	clip( PlaneDists );
	float Alpha = min( min( PlaneDists.x, PlaneDists.y ), 1 );

#ifndef _PC_TARGET
	[branch]
#endif
	if (Alpha > 0.001)
	{
		// Get lighting parameters
		float kSpecularPower = PSParams.x ;
		float kEdgeAlpha = PSParams.y;	
		float kTranslucency = PSParams.z;
		float kTranslucencySpread = PSParams.w;	
		float kSelfShadowing = PSParams2.x * 0.3f;

		// Read colours

		float2 HairTexCoord = float2( 0.5 + 0.5*In.Params.y, 1-In.Params.x );
		float4 SurfaceTexCoord = float4( In.Params2.x, 1.0-In.Params2.y, 0, 0 );

#ifdef _FEATHERS
	
		float4 HairColour			= tex2D( diffuse_tex, HairTexCoord );
		float4 HairDiffuse			= float4( lerp( kBaseColour, kTipColour, In.Params.x ), 1 );
		float4 HairAmbient			= float4( lerp( kBaseAmbientColour, kTipAmbientColour, In.Params.x ), 1 );
		float3 HairSpecular			= tex2D(specular_tex, HairTexCoord).rgb;
	
		clip( HairColour.a - 1.0/256.0 );

		float4 SurfaceColour		= tex2Dlod( surf_diff_tex, SurfaceTexCoord );
		float3 SurfaceSpecular		= tex2Dlod(surf_spec_tex, SurfaceTexCoord).rgb;

		float4 DiffuseColour		= SurfaceColour * HairColour * HairDiffuse;
		float4 AmbientColour		= SurfaceColour * HairColour * HairAmbient;
		float3 SpecularColour		= SurfaceSpecular * HairSpecular * float4( kInputSpecularColour, 1 ) * 1.5; // Note the 1.5x boost
		float3 DiffuseEnvSpecTex    = In.Params.x;
#else

		float4 TexA					= tex2Dlod( base_diff_tex, SurfaceTexCoord );
		float4 TexB					= tex2Dlod( tip_diff_tex, SurfaceTexCoord );

		float4 HairColour			= float4( lerp( TexA.rgb, TexB.rgb, In.Params.x ), 1 );
		float4 HairDiffuse			= float4( lerp( kBaseColour, kTipColour, In.Params.x ), 1 );
		float4 HairAmbient			= float4( lerp( kBaseAmbientColour, kTipAmbientColour, In.Params.x ), 1 );
		float3 HairSpecularTex      = tex2D(specular_tex, HairTexCoord).rgb;
		float3 HairSpecular			= HairSpecularTex * kInputSpecularColour;
		float3 DiffuseEnvSpecTex	= HairSpecularTex;

		float4 SurfaceColour		= float4(1,1,1,1);
		float3 SurfaceSpecular		= tex2Dlod(surf_spec_tex, SurfaceTexCoord).rgb;

		float4 DiffuseColour		= SurfaceColour * HairColour * HairDiffuse;
		float4 AmbientColour		= SurfaceColour * HairColour * HairAmbient;
		float3 SpecularColour		= SurfaceSpecular * HairSpecular;
#endif

		// Normalise basis vectors
		float3 Tangent = normalize(In.Tangent);
		float3 View = normalize(In.View);
		float3 Normal = normalize(In.Nrm);
//		float3 Binormal = normalize(cross(Tangent, Normal));
		float3 Binormal = normalize(In.Bin.xyz);

#ifdef _FEATHERS
		// Perform 'tangent-mapping'
//		float NormTex = tex2D(normalmap_tex, float2(HairTexCoord.x, HairTexCoord.y)).r;
//		float BinormalDir = NormTex * 2.0 - 1.0;
		float BinormalDir = HairTexCoord.x * 0.6 - 0.3;
		float TangentDir = 1.0 - abs(BinormalDir);
		float invLen = 1.0 / sqrt(BinormalDir * BinormalDir + TangentDir * TangentDir);
		Tangent = (Tangent * TangentDir + Binormal * BinormalDir) * invLen;
#endif
		
		// Calculate the diffuse environment lighting contributions
		float diffEnvFresnel = pow(abs(1.0 - abs(dot(View, Normal))), PSParams2.a);
		float3 worldNormal = mul((float3x3)envMtx, Normal);
		float3 worldEnvVec = mul((float3x3)envMtx, reflect(View, Normal));

		float3 diffuseEnvLightDiff = texCUBE(diffenv_tex, worldNormal).rgb * PSParams2.g * DiffuseEnvSpecTex; // Note: Uses per-hair specular map only (?)
		float3 diffuseEnvLightSpec = texCUBE(diffenv_tex, worldEnvVec).rgb * PSParams2.b * diffEnvFresnel;

		// Calculate lighting

		// Invert the lighting if required (for reflections)
		float3 lp0 = lightPosition0.xyz * fs_lightingAdjust;
		float3 lp1 = lightPosition1.xyz * fs_lightingAdjust;
		float3 lp2 = lightPosition2.xyz * fs_lightingAdjust;
		float3 lp0S = specLightPosition0.xyz * fs_lightingAdjust;
		float3 lp1S = specLightPosition1.xyz * fs_lightingAdjust;
		float3 lp2S = specLightPosition2.xyz * fs_lightingAdjust;
	
		// Calculate diffuse contribution
		float3 ldotb = float3(dot(Tangent, lp0), dot(Tangent, lp1), dot(Tangent, lp2));
		float3 ad = sqrt(1.0 - (ldotb * ldotb));
		float3 ldotn = float3(dot(Normal, lp0), dot(Normal, lp1), dot(Normal, lp2));	

		// Doing the self-shadowing adjustment this way fixes the dark banding down the centre of the feathers.
		float3 anisoDiffuse = clamp(ldotn * ad * kSelfShadowing + 1.0, 0.0, 1.0);

		float3 DiffuseLight = diffuseEnvLightDiff + (lightColor0 * anisoDiffuse.x + lightColor1 * anisoDiffuse.y + lightColor2 * anisoDiffuse.z);

		// Calculate specular contribution
		float3 ldotbS = float3(dot(Tangent, lp0S), dot(Tangent, lp1S), dot(Tangent, lp2S));
		float3 adS = sqrt(1.0 - (ldotbS * ldotbS));
		float3 vdotb = dot(View, Tangent);
		float3 vbf = sqrt(1.0 - (vdotb * vdotb));
		float3 anisoSpecular = adS * vbf - (ldotbS * vdotb);
		anisoSpecular = pow(max(0.0, anisoSpecular), kSpecularPower);

		anisoSpecular *= max(0.0, ldotn);
		float3 SpecularLight = (specLightColor0 * anisoSpecular.x) + (specLightColor1 * anisoSpecular.y) + (specLightColor2 * anisoSpecular.z);
		SpecularLight += diffuseEnvLightSpec;

		// Calculate overall surface colour
		float3 FinalSurfaceColour = (DiffuseColour.rgb * DiffuseLight.rgb) + (SpecularColour * SpecularLight) + (AmbientColour.rgb * sceneAmbientColor);

		float EdgeAlpha = 1.0f;
#ifdef _FUR
		// Calculate 'edge' parameters for varying colour and alpha over the width of the hair
		float edge = min( abs(In.Params.y), 1.0 );
		float edge1 = 1.0 - pow(edge,kEdgeAlpha); // Edge value used for alpha
		float edge2 = pow(edge,kTranslucencySpread); // Edge value used for lighting (translucency)
		EdgeAlpha *= In.Params.w * edge1;
		//FinalSurfaceColour *= 1.0 + edge2*kTranslucency; // Add translucency
#endif

		// REFLECTION_CODE (Below)
		// Applies a fade to green on geometry with negative y - presumed to be underfloor reflections (for TT Animation).
		float3 fadeGreen = float3(0.162, 0.397, 0.078);
		float reflectionFade = saturate((PSReflectParams.x + worldY) / PSReflectParams.x);
		float fadeFactor = max(step(0.0, worldY), PSReflectParams.z * pow(reflectionFade, PSReflectParams.y));
		FinalSurfaceColour.rgb = lerp(fadeGreen, FinalSurfaceColour.rgb, fadeFactor);
		
		// Output colour
		fout.Colour.rgb = FinalSurfaceColour.rgb;
		fout.Colour.a = clamp( EdgeAlpha * DiffuseColour.a, 0, 1 ) * Alpha;
	
		// Normal			
		fout.normal.rg = Normal.xy * 0.5 + 0.5;
		// Specular intensity (for deferred light) & 'don't self shadow' flag
		fout.normal.z = 1.0;
		// Glow factor
		fout.normal.w = 0.0;

		// Aldebo
		fout.albedo.rgb = DiffuseColour.rgb;

		// Specular power & 'receive shadow' flag
		fout.albedo.a = 1;

		// Replicate the RT0.a onto the other targets to get correct blending.
		fout.normal.a = 1;//fout.Colour.a;
		fout.albedo.a = 1;//fout.Colour.a;

#ifdef _PC_TARGET
		fout.depth = ScreenPos.z;
#endif

		return fout;
	}
	else
	{
		fout.Colour.rgb = float3(1,1,1);
		fout.Colour.a = 0.0;
		fout.normal.rgb = float3(0.0,0.0,0.0);
		fout.normal.w = 1.0;
		fout.albedo = float4(0,0,0,0);
#ifdef _PC_TARGET
		fout.depth = ScreenPos.z;
#endif
		return fout;
	}
}

#ifdef _PC_TARGET
FragmentOutput surfgeomdepth_px( VS_OUT In )
{
	FragmentOutput fout;

	// Clip out any normal geometry passing downwards through the floor, and also any reflected geometry passing upwards through the floor.
	float worldY = In.Bin.w;
	clip(worldY * fs_lightingAdjust.y);

#ifdef _FEATHERS
	float2 HairTexCoord = float2( 0.5 + 0.5*In.Params.y, 1-In.Params.x );
	float Alpha = tex2D( diffuse_tex, HairTexCoord ).a - 1.0/256.0;
	clip( Alpha );
#endif

	float Depth = In.Screen.z / In.Screen.w;
	Depth = fs_projection_params.x / (1.0 - Depth*fs_projection_params.w);
	fout.Colour.rgba = Depth * 0.001;
	fout.normal.rgba = 0;
	fout.albedo = 0;
	fout.depth = 0;
	return fout;
}

#else

FragmentOutput surfgeomdepth_px( VS_OUT In )
{
	FragmentOutput fout;
	fout.Colour.rgb = float3(1,1,1);
	fout.Colour.a = 0.0f;
	fout.normal.rg = normalize(In.Nrm).xy * 0.5 + 0.5;
	fout.normal.z = 0.0;
	fout.normal.w = 1.0;
	fout.albedo = float4(0,0,0,0);
	return fout;
}

#endif
